

/**
 ******************************************************************************
 *
 * @file        MG04_06_Control.c
 * @brief       
 
 * @par         Project
 *              MG32x02z
 * @version     V1.01
 * @date        2022/08/01
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */
 

/* Includes ------------------------------------------------------------------*/
#include "MG04_06_Drama.h"
#include "MG04_06_Global.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static uint16_t const MG04_06_LCDColor[ARGB_Default_Color] = {RED,YELLOW,GREEN,CYAN,BLUE,MAGENTA,WHITE};   /*!< Mapping LCD color and ARGB default color talbe.*/


/* Private function prototypes -----------------------------------------------*/
static void API_LCD_DisplayARGModeString(void);
static void API_LCD_DisplayARGBColor(void);
static void API_LCDDynamicFormChange( uint8_t  LCD_Page);

static void API_ARGB_ChangeColor(void);
static void API_ARGB_AutoMode_Init(void);
static void API_ARGB_ChangeMode(uint8_t ARG_ChangeMode);

static void API_KeyFunction0_Handle(void);
static void API_KeyFunction1_Handle(void);
static void API_KeyFunction2_Handle(void);
static void API_ESCFunction_Handle(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/



/**
 *******************************************************************************
 * @brief	    BSP initial callback function.
 * @details     Initial MG04-06 common control parameter.
 * @return      
 * @exception   No
 * @note        
 *******************************************************************************
 */
void BSP_InitCallback(void)
{
    /*Key Common Parameter Initial*/
    MG04_06_CTR.KeyStatus_ChangeFlag = 0;
    
    memset(MG04_06_CTR.KeyStatus,0,MG04_06_TOTAL_KEY);
    
    /*Time Base Parameter Initial*/
    MG04_06_CTR.TimeBaseCount = MG04_06_TIMEHANDLE_BACKGROUND;
    MG04_06_CTR.TimeBase_Flag = 1;
    
    
    /*The Other Parameter Initial*/
    MG04_06_CTR.MainState                 = MG04_06_STATE_MAIN_FRONTPAGE;
    MG04_06_CTR.SubState                  = 0;
    MG04_06_CTR.SubStateBefore            = 0;
    MG04_06_CTR.IdleTimeCompare           = 0;                                /*!< Default is Disable*/
    MG04_06_CTR.IdleTime                  = 0; 
    MG04_06_CTR.IdleState                 = MG04_06_IDLESTATE_NO;    
    MG04_06_CTR.SubStateLock              = DISABLE;
    MG04_06_CTR.ESC_FunctionFlag          = 0;
     
}    
    

/**
 * @name  LCD handler
 */
///@{
/**
 *******************************************************************************
 * @brief	  LCD initial callback function.  
 * @details     
 * @return      
 * @exception No
 * @note        
 *******************************************************************************
 */
void BSP_LCD_InitCallback(void)
{

    /*LCD Parameter Initial*/
    MG04_06_CTR.pLCDframeTable            = MG04_06_MainButtonFrame;
    MG04_06_CTR.pLCDEnterTable            = MG04_06_MainEnterDrama;
    MG04_06_CTR.LCD_DynamicUpdateFlag     = LCD_DYNAMICUPDATE_OPITIONFRAME;
    MG04_06_CTR.LCD_BackgroundColor       = GRAY;
    
    MG04_06_CTR.LCD_Status                = Idle;
    MG04_06_CTR.LCD_BackgroundUpdateFlag  = 1;
    
}
/**
 *******************************************************************************
 * @brief	  Callback function before LCD update background from flash(MX25R6435F)
 * @details     
 * @return      
 * @exception No
 * @note        
 *******************************************************************************
 */
void BSP_LCD_UpdateFromFlashCallback(void)
{
    MG04_06_CTR.LCD_Status = Busy;
}
/**
 *******************************************************************************
 * @brief	  Callback function after LCD update background complete 
              from flash(MX25R6435F)
 * @details     
 * @return      
 * @exception No
 * @note        
 *******************************************************************************
 */
void BSP_LCD_UpdateFromFlashCpltCallback(void)
{
    MG04_06_CTR.LCD_Status = Idle;
}

/**
 *******************************************************************************
 * @brief	  LCD update background function
 * @details     
 * @return      
 * @exception No
 * @note        
 *******************************************************************************
 */
void API_LCD_BackgroundUpdate(void)
{
    if( MG04_06_CTR.LCD_Status != Idle || MG04_06_CTR.LCD_BackgroundUpdateFlag == 0)
    {
        return;
    }

    MG04_06_CTR.LCD_Status = Busy;
    MG04_06_CTR.LCD_BackgroundUpdateFlag = 0;


    if( MG04_06_CTR.MainState == MG04_06_STATE_MAIN_FRONTPAGE)
    {
        BSP_LCD_FillMonochrome(&LCD_CTR, 0, 0, 239, 319, BLACK);

        BSP_LCD_DefineFormatUpdate( 0, GRAY, WHITE, (uint8_t *)"ARGB LED");
        BSP_LCD_DefineFormatUpdate( 5, GRAY, WHITE, (uint8_t *)"Demo");
        BSP_LCD_DefineFormatUpdate( 6, GRAY, WHITE, (uint8_t *)"Set");
    }
    else if( MG04_06_CTR.MainState == MG04_06_STATE_MAIN_SETTINGPAGE)
    {
        BSP_LCD_FillMonochrome(&LCD_CTR, 0, 0, 239, 319, BLACK);
        BSP_LCD_DefineFormatUpdate( 0, GRAY, WHITE, (uint8_t *)"LCD Off(sec):");
        BSP_LCD_DefineFormatUpdate( 5, GRAY, WHITE, (uint8_t *)"Back");
    }
    else
    {
        BSP_LCD_UpdateFromFlash(LCD_BackGroundFlashTable[ MG04_06_CTR.MainState - 1],LCD_FLASH_BACKGROUND_SIZE);
        return;
    }
    
    BSP_LCD_ShowString(&LCD_CTR,107,   2, 132, 24, GBLUE, (uint8_t *)"MG32F02V032");
    BSP_LCD_ShowString(&LCD_CTR, 59,  26, 180, 24, WHITE, (uint8_t *)"ARGB Demo Board");
    
    MG04_06_CTR.LCD_Status = Idle;

}

/**
 *******************************************************************************
 * @brief	  Change LCD display when user operate.
 * @details     
 * @return      
 * @exception No
 * @note        
 *******************************************************************************
 */
void API_LCD_DynamicUpdate(void)
{
    uint8_t   API_LCD_DynamicUpdateTmp;
    uint16_t  API_LCD_DynamicUpdateTmp2;

    if( MG04_06_CTR.LCD_Status != Idle)
    {
        return;
    }
    if( MG04_06_CTR.LCD_DynamicUpdateFlag == 0)
    {
        return;
    }
    

    if( (MG04_06_CTR.LCD_DynamicUpdateFlag & LCD_DYNAMICUPDATE_OPITIONFRAME) != 0)
    {
        MG04_06_CTR.LCD_DynamicUpdateFlag &= (~LCD_DYNAMICUPDATE_OPITIONFRAME);
        
        
        API_LCD_DynamicUpdateTmp = MG04_06_CTR.SubStateBefore * 5;
        
        /*Opition frame upadate*/
        BSP_LCD_ButtonUp(&LCD_CTR, \
                          MG04_06_CTR.pLCDframeTable[API_LCD_DynamicUpdateTmp + 0],\
                          MG04_06_CTR.pLCDframeTable[API_LCD_DynamicUpdateTmp + 1],\
                          MG04_06_CTR.pLCDframeTable[API_LCD_DynamicUpdateTmp + 2],\
                          MG04_06_CTR.pLCDframeTable[API_LCD_DynamicUpdateTmp + 3],\
                          MG04_06_CTR.LCD_BackgroundColor);
        
            
        API_LCD_DynamicUpdateTmp = MG04_06_CTR.SubState * 5;
        
        BSP_LCD_ButtonUp(&LCD_CTR, \
                          MG04_06_CTR.pLCDframeTable[API_LCD_DynamicUpdateTmp + 0],\
                          MG04_06_CTR.pLCDframeTable[API_LCD_DynamicUpdateTmp + 1],\
                          MG04_06_CTR.pLCDframeTable[API_LCD_DynamicUpdateTmp + 2],\
                          MG04_06_CTR.pLCDframeTable[API_LCD_DynamicUpdateTmp + 3],\
                          MG04_06_CTR.pLCDframeTable[API_LCD_DynamicUpdateTmp + 4]);
    }
    
    /*The other Update*/
    if( (MG04_06_CTR.LCD_DynamicUpdateFlag & LCD_DYNAMICUPDATE_OTHER) == 0)
    {
        return;
    }
    
    MG04_06_CTR.LCD_DynamicUpdateFlag &= (~LCD_DYNAMICUPDATE_OTHER); 
    
    switch( MG04_06_CTR.MainState)
    {
        case MG04_06_STATE_MAIN_FRONTPAGE:
                                              BSP_LCD_FillMonochrome(&LCD_CTR, 10, 206, 42, 222, LCD_CTR.BackColor);
                                              if( MG04_06_CTR.ARGB_Auto == ENABLE)
                                              {
                                                  BSP_LCD_DefineFormatStringUpdate( 5, YELLOW, (uint8_t *)"Demo");

                                              }
                                              else
                                              {
                                                  BSP_LCD_DefineFormatStringUpdate( 5, WHITE, (uint8_t *)"Demo");
                                              }
                                              break;
        case MG04_06_STATE_MAIN_ARGBFRONTPAGE:
                                              if( MG04_06_CTR.MainState == MG04_06_STATE_MAIN_ARGBFRONTPAGE)
                                              {
                                                  MG04_06_CTR.LCD_DynamicUpdateFlag &= (~LCD_DYNAMICUPDATE_OTHER);
                                                  
                                                  API_LCD_DisplayARGModeString();
                                                  API_LCD_DisplayARGBColor();
                                                  return;
                                              }    
                                              break;
        case MG04_06_STATE_MAIN_SETTINGPAGE:
                                              API_LCD_DynamicUpdateTmp  = MG04_06_CTR.SubState * 5; 
                                              API_LCD_DynamicUpdateTmp2 = (uint16_t)(MG04_06_CTR.IdleTimeCompare / 1000);
                                              BSP_LCD_FillMonochrome(&LCD_CTR, 170, 65, 230, 90, LCD_CTR.BackColor);
                                              
        
                                              if( MG04_06_CTR.SubStateLock == ENABLE)
                                              {
                                                  BSP_LCD_DefineFormatUpdate(0,RED,WHITE,(uint8_t*)"");
                                              }
                                              else
                                              {
                                                  BSP_LCD_DefineFormatUpdate(0,MG04_06_CTR.pLCDframeTable[API_LCD_DynamicUpdateTmp + 4],WHITE,(uint8_t*)"");
                                              }
        
                                              if( API_LCD_DynamicUpdateTmp2 !=0)
                                              {
                                                  BSP_LCD_ShowChar(&LCD_CTR,175,65,(0x30 | (uint8_t)API_LCD_DynamicUpdateTmp2),1,WHITE);
                                                  BSP_LCD_ShowChar(&LCD_CTR,183,65,0x30,1,WHITE);
                                              }
                                              else
                                              {
                                                  BSP_LCD_ShowString(&LCD_CTR,175,65,232,24,WHITE,(uint8_t*)"Disable");
                                              }
                                              break;
        default:
                                              break;
        
    }
}

/**
 *******************************************************************************
 * @brief	  LCD display ARGB mode in ARGB front page.
 * @details     
 * @return      
 * @exception No
 * @note        
 *******************************************************************************
 */
static void API_LCD_DisplayARGModeString(void)
{
    if( MG04_06_CTR.ARGB_Auto == ENABLE)
    {
        BSP_LCD_ShowString(&LCD_CTR, 77,  23, 232, 24, YELLOW, (uint8_t *)"AUTO Mode");
        return;
    }
    
    switch( (ARGB_CTR.DramaMode & ARGB_MODE_MASK)) 
    {
        case ARGB_MODE_DISABLE:
                                    BSP_LCD_ShowString(&LCD_CTR, 77,  23, 232, 24, YELLOW, (uint8_t *)"DISABLE");
                                    break;
        case ARGB_MODE_SINGLE_COLOR:
                                    BSP_LCD_ShowString(&LCD_CTR, 77,  23, 232, 24, YELLOW, (uint8_t *)"SINGLE COLOR Mode");
                                    break;
        case ARGB_MODE_CROSS:
                                    BSP_LCD_ShowString(&LCD_CTR, 77,  23, 232, 24, YELLOW, (uint8_t *)"CROSS Mode");
                                    break;
        case ARGB_MODE_NET:
                                    BSP_LCD_ShowString(&LCD_CTR, 77,  23, 232, 24, YELLOW, (uint8_t *)"NET Mode");
                                    break;
        case ARGB_MODE_STRING:
                                    BSP_LCD_ShowString(&LCD_CTR, 77,  23, 232, 24, YELLOW, (uint8_t *)"STRING Mode");
                                    break;
        default:
                                    break;
    }
}
/**
 *******************************************************************************
 * @brief	  In ARGB front page LCD display that ARGB color in this ARG mode.
 * @details     
 * @return      
 * @exception No
 * @note        
 *******************************************************************************
 */
static void API_LCD_DisplayARGBColor(void)
{  
    uint16_t API_LCD_DisplayARGBColorTmp;
    uint8_t  API_LCD_DisplayARGBColorTmp2;
    
    if( MG04_06_CTR.ARGB_Auto == ENABLE || ((ARGB_CTR.DramaMode & ARGB_MODE_MASK)==ARGB_MODE_DISABLE))
    {
        BSP_LCD_FillMonochrome(&LCD_CTR, 77, 50, 97, 70, DARKGREEN);
        BSP_LCD_FillMonochrome(&LCD_CTR,107, 50,127, 70, DARKGREEN);
        
        BSP_LCD_ShowString(&LCD_CTR, 77,  54, 232, 24, YELLOW, (uint8_t *)"NO");
    }
    else
    {
        API_LCD_DisplayARGBColorTmp2 = MG04_06_CTR.ARGBMode_Index * 2;
        API_LCD_DisplayARGBColorTmp = MG04_06_LCDColor[MG04_06_CTR.ARGBColor_Opition[API_LCD_DisplayARGBColorTmp2]];
        BSP_LCD_FillMonochrome(&LCD_CTR, 77, 50, 97, 70, API_LCD_DisplayARGBColorTmp );
        if( ARGB_CTR.ColorNumber == 2)
        {
            API_LCD_DisplayARGBColorTmp = MG04_06_LCDColor[MG04_06_CTR.ARGBColor_Opition[(API_LCD_DisplayARGBColorTmp2+1)]];
            BSP_LCD_FillMonochrome(&LCD_CTR,107, 50,127, 70, API_LCD_DisplayARGBColorTmp);
        }
    }
}

/**
 *******************************************************************************
 * @brief	    LCD form mapping function
 * @details     
 * @return      
 * @exception   No
 * @note            
 *******************************************************************************
 */
static void API_LCDDynamicFormChange( uint8_t  LCD_Page)
{
    MG04_06_CTR.LCD_BackgroundColor = DARKGREEN;
    
    switch( LCD_Page)
    {
        case MG04_06_STATE_MAIN_FRONTPAGE:
                                                MG04_06_CTR.pLCDframeTable      = MG04_06_MainButtonFrame;
                                                MG04_06_CTR.pLCDEnterTable      = MG04_06_MainEnterDrama;
                                                break;
        case MG04_06_STATE_MAIN_ARGBFRONTPAGE:   
                                                MG04_06_CTR.pLCDframeTable      = MG04_06_ARGBMainButtonFrame;
                                                MG04_06_CTR.pLCDEnterTable      = MG04_06_ARGBMainEnterDrama;
                                                                                
                                                return;                          
        case MG04_06_STATE_MAIN_ARGBMODEPAGE:                                   
                                                MG04_06_CTR.pLCDframeTable      = MG04_06_ARGBModeButtonFrame;
                                                MG04_06_CTR.pLCDEnterTable      = MG04_06_ARGBModeEnterDrama; 
                                                return;                          
        case MG04_06_STATE_MAIN_ARGB2COLORPAGE:                                 
                                                MG04_06_CTR.pLCDframeTable      = MG04_06_ARGB2ColorButtonFrame;
                                                MG04_06_CTR.pLCDEnterTable      = MG04_06_ARGB2ColorEnterDrama;
                                                return;                          
        case MG04_06_STATE_MAIN_ARGBCOLORPAGE:                                  
                                                MG04_06_CTR.pLCDframeTable      = MG04_06_ARGBColorButtonFrame;
                                                MG04_06_CTR.pLCDEnterTable      = MG04_06_ARGBColorEnterDrama;
                                                return;                          
        case MG04_06_STATE_MAIN_SETTINGPAGE:                                    
                                                MG04_06_CTR.pLCDframeTable      = MG04_06_SettingButtonFrame;
                                                MG04_06_CTR.pLCDEnterTable      = MG04_06_SettingEnterDrama;
                                                break;
        default:                                
                                                return;         
    }
    MG04_06_CTR.LCD_BackgroundColor = GRAY;
}

///@}


/**
 * @name  ARGB handler
 */
///@{
/**
 *******************************************************************************
 * @brief	    ARGB initial callback function.
 * @details     Initial BSP_ARGB control parameter.
 * @return      
 * @exception   No
 * @note        
 *******************************************************************************
 */
void BSP_ARGB_InitCallback(void)
{
    uint8_t BSP_ARGB_InitCallbackTmp;
    uint8_t BSP_ARGB_InitCallbackTmp2;
    
    /*Default initial color setting for every ARGB mode*/
    for(BSP_ARGB_InitCallbackTmp = 0; BSP_ARGB_InitCallbackTmp < (ARGB_MODE_TOTAL-2); BSP_ARGB_InitCallbackTmp++)
    {
        BSP_ARGB_InitCallbackTmp2 = BSP_ARGB_InitCallbackTmp * 2;
        
        MG04_06_CTR.ARGBColor_Opition[BSP_ARGB_InitCallbackTmp2    ] = 0;
        MG04_06_CTR.ARGBColor_Opition[BSP_ARGB_InitCallbackTmp2 + 1] = 3;
    }
    
    MG04_06_CTR.ARGBMode_Index  = 0;                  
    MG04_06_CTR.ARGBColor_Index = 0;                  
}

/**
 *******************************************************************************
 * @brief	    ARGB Handler (include ARGB auto mode)
 * @details     
 * @return      
 * @exception   No
 * @note            
 *******************************************************************************
 */
void API_ARGBDrama_Handle(void)
{
    uint8_t API_ARGBDrama_HandleTmp;
    
    BSP_ARGB_Drama();
    
    /*ARGB Normal Mode*/
    if( MG04_06_CTR.ARGB_Auto == DISABLE)
    {
        return;
    }
    /*ARGB Auto Mode*/
    

    if( MG04_06_CTR.ARGB_AUTO_DramaMainCount != ARGB_AUTO_DRAMA_SINGLE_COLOR)
    {
        if((ARGB_CTR.DramaMode & ARGB_ACT_MASK) != 0)
        {
            return;
        }
        MG04_06_CTR.ARGB_AUTO_DramaMainCount ++;
        
        API_ARGBDrama_HandleTmp = MG04_06_CTR.ARGB_AUTO_DramaMainCount * 3;
        
        BSP_ARGB_ChangeColor( 0, ARGB_DefaultColor[ API_ARGBDrama_HandleTmp + ARGB_R_Color],
                                 ARGB_DefaultColor[ API_ARGBDrama_HandleTmp + ARGB_G_Color],
                                 ARGB_DefaultColor[ API_ARGBDrama_HandleTmp + ARGB_B_Color]);

        BSP_ARGB_ChangeColor( 1, ARGB_DefaultColor[ 0 + ARGB_R_Color],
                                 ARGB_DefaultColor[ 0 + ARGB_G_Color],
                                 ARGB_DefaultColor[ 0 + ARGB_B_Color]);
        
        switch( MG04_06_CTR.ARGB_AUTO_DramaMainCount)
        {
            case ARGB_AUTO_DRAMA_NET:
                                        BSP_ARGB_ChangeDrama( ARGB_MODE_NET,DISABLE);
                                        return;
            case ARGB_AUTO_DRAMA_STRING:
                                        BSP_ARGB_ChangeDrama( ARGB_MODE_STRING,DISABLE);
                                        return;
            default:
                                        MG04_06_CTR.ARGB_AUTO_DramaMainCount = ARGB_AUTO_DRAMA_SINGLE_COLOR;
                                        break;        
        }
    }
    
    if( MG04_06_CTR.ARGB_AUTO_Count != MG04_06_CTR.ARGB_AUTO_CountCompare)
    {
        MG04_06_CTR.ARGB_AUTO_Count = MG04_06_CTR.ARGB_AUTO_Count + 1;
        return;
    }
    
    MG04_06_CTR.ARGB_AUTO_Count = 0;
    
    //ARGB Singal Color Display.

    if( MG04_06_CTR.ARGB_AUTO_DramaSubCount == MG04_06_CTR.ARGB_AUTO_DramaSubCountCompare)
    {
        MG04_06_CTR.ARGB_AUTO_DramaMainCount = ARGB_AUTO_DRAMA_CROSS;
        MG04_06_CTR.ARGB_AUTO_DramaSubCount  = 0;
        
        BSP_ARGB_ChangeColor( 0, ARGB_DefaultColor[ 3 + ARGB_R_Color],
                                 ARGB_DefaultColor[ 3 + ARGB_G_Color],
                                 ARGB_DefaultColor[ 3 + ARGB_B_Color]);
        
        BSP_ARGB_ChangeDrama( ARGB_MODE_CROSS,DISABLE);
    }
    else
    {
        MG04_06_CTR.ARGB_AUTO_DramaSubCount ++;
        
        API_ARGBDrama_HandleTmp = MG04_06_CTR.ARGB_AUTO_DramaSubCount * 3;
    
        BSP_ARGB_ChangeColor( 0, ARGB_DefaultColor[API_ARGBDrama_HandleTmp + ARGB_R_Color],
                                 ARGB_DefaultColor[API_ARGBDrama_HandleTmp + ARGB_G_Color],
                                 ARGB_DefaultColor[API_ARGBDrama_HandleTmp + ARGB_B_Color]);
        
        BSP_ARGB_ChangeDrama( ARGB_MODE_SINGLE_COLOR,DISABLE);
    }
}

/**
 *******************************************************************************
 * @brief	  Change ARGB Color
 * @details     
 * @return      
 * @exception No
 * @note        
 *******************************************************************************
 */
static void API_ARGB_ChangeColor(void)
{
    uint8_t API_ARGB_ChangeColorTmp;
    
    if( ARGB_CTR.DramaMode == ARGB_MODE_DISABLE)
    {
        return;
    }
    
    API_ARGB_ChangeColorTmp = MG04_06_CTR.SubState * 3;
    
    
    ARGB_CTR.R_Color[MG04_06_CTR.ARGBColor_Index] = ARGB_DefaultColor[ API_ARGB_ChangeColorTmp + ARGB_R_Color];
    ARGB_CTR.G_Color[MG04_06_CTR.ARGBColor_Index] = ARGB_DefaultColor[ API_ARGB_ChangeColorTmp + ARGB_G_Color];
    ARGB_CTR.B_Color[MG04_06_CTR.ARGBColor_Index] = ARGB_DefaultColor[ API_ARGB_ChangeColorTmp + ARGB_B_Color];
    
    
    BSP_ARGB_ChangeDrama( (ARGB_CTR.DramaMode & ARGB_MODE_MASK),ENABLE);
   

    API_ARGB_ChangeColorTmp = (MG04_06_CTR.ARGBMode_Index * 2) + MG04_06_CTR.ARGBColor_Index;
    MG04_06_CTR.ARGBColor_Opition[API_ARGB_ChangeColorTmp] = MG04_06_CTR.SubState;
}

/**
 *******************************************************************************
 * @brief	    ARGB initial in ARGB mode is auto mode.
 * @details     
 * @return      
 * @exception   No
 * @note            
 *******************************************************************************
 */
static void API_ARGB_AutoMode_Init(void)
{
    MG04_06_CTR.ARGB_AUTO_Count                = 0;
    MG04_06_CTR.ARGB_AUTO_CountCompare         = 100;                                           
    MG04_06_CTR.ARGB_AUTO_DramaMainCount       = ARGB_AUTO_DRAMA_SINGLE_COLOR;
    MG04_06_CTR.ARGB_AUTO_DramaMainCountCompare= ARGB_AUTO_DRAMA_STRING;
    MG04_06_CTR.ARGB_AUTO_DramaSubCount        = 0;
    MG04_06_CTR.ARGB_AUTO_DramaSubCountCompare = (ARGB_Default_Color - 1);

    BSP_ARGB_ChangeColor( 0, ARGB_DefaultColor[ 0 + ARGB_R_Color],
                             ARGB_DefaultColor[ 0 + ARGB_G_Color],
                             ARGB_DefaultColor[ 0 + ARGB_B_Color]);
    
    
    
    MG04_06_CTR.ARGB_Auto = ENABLE;
    
    BSP_ARGB_ChangeDrama( ARGB_MODE_SINGLE_COLOR,DISABLE);
}

/**
 *******************************************************************************
 * @brief	    Change ARGB mode.
 * @details     
 * @return      
 * @exception   No
 * @note            
 *******************************************************************************
 */
static void API_ARGB_ChangeMode(uint8_t ARG_ChangeMode)
{
    uint32_t API_ARGB_ChangeModeTmp;
    uint16_t API_ARGB_ChangeModeColorTmp;
    uint16_t API_ARGB_ChangeModeColor2Tmp;
    
    
    if( ARG_ChangeMode == ARGB_AUTO_MODE)
    {
        API_ARGB_AutoMode_Init();
        return;
    }
    
    MG04_06_CTR.ARGB_Auto  = DISABLE;
    API_ARGB_ChangeModeTmp = (ARG_ChangeMode <<ARGB_MODE_SHIFT) & ARGB_MODE_MASK;
    
    if( API_ARGB_ChangeModeTmp != ARGB_MODE_DISABLE)
    {
        MG04_06_CTR.ARGBMode_Index   =  ( ARG_ChangeMode - 1); 
        
        API_ARGB_ChangeModeColorTmp  = ( MG04_06_CTR.ARGBMode_Index * 2);
        
        
        API_ARGB_ChangeModeColor2Tmp = MG04_06_CTR.ARGBColor_Opition[ API_ARGB_ChangeModeColorTmp + 1];
        API_ARGB_ChangeModeColorTmp  = MG04_06_CTR.ARGBColor_Opition[ API_ARGB_ChangeModeColorTmp];
        
        
        API_ARGB_ChangeModeColorTmp = API_ARGB_ChangeModeColorTmp * 3;
        
        BSP_ARGB_ChangeColor(0, ARGB_DefaultColor[API_ARGB_ChangeModeColorTmp + ARGB_R_Color],
                                ARGB_DefaultColor[API_ARGB_ChangeModeColorTmp + ARGB_G_Color],
                                ARGB_DefaultColor[API_ARGB_ChangeModeColorTmp + ARGB_B_Color]);

        
        API_ARGB_ChangeModeColor2Tmp = API_ARGB_ChangeModeColor2Tmp * 3;
        
        BSP_ARGB_ChangeColor(1, ARGB_DefaultColor[API_ARGB_ChangeModeColor2Tmp + ARGB_R_Color],
                                ARGB_DefaultColor[API_ARGB_ChangeModeColor2Tmp + ARGB_G_Color],
                                ARGB_DefaultColor[API_ARGB_ChangeModeColor2Tmp + ARGB_B_Color]);

        
    }
    switch( API_ARGB_ChangeModeTmp)
    {
        case ARGB_MODE_CROSS:
        case ARGB_MODE_NET:
        case ARGB_MODE_STRING:
                              BSP_ARGB_ChangeDrama( API_ARGB_ChangeModeTmp,ENABLE);
                              break;
        default:
                              BSP_ARGB_ChangeDrama( API_ARGB_ChangeModeTmp,DISABLE);
                              break;
    }
}
///@}

/**
 * @name  Key handler
 */
///@{
/**
 *******************************************************************************
 * @brief	    Touch key initial callback function.
 * @details     Initial BSP_ARGB control parameter.
 * @return      
 * @exception   No
 * @note        
 *******************************************************************************
 */
void BSP_TouchKey_InitCallback(void)
{
    /*Touch Key Parameter Initial*/
    MG04_06_CTR.Touchkey_State        = TOUCHKEY_KEY0_CHARGE;
    MG04_06_CTR.TouchKey_BeforeStatus = 0;
    MG04_06_CTR.TouchKey_NowStatus    = 0;
}
/**
 *******************************************************************************
 * @brief	    SARADC key initial callback function.
 * @details     Initial BSP_SARADC control parameter.
 * @return      
 * @exception   No
 * @note        
 *******************************************************************************
 */
void BSP_SARADCKey_InitCallback(void)
{
    /*KEY (SARADC Button) Parameter Initial*/
    MG04_06_CTR.SKey_BeforeStatus    = 0;
    MG04_06_CTR.SKey_NowStatus       = 0;
}
/**
 *******************************************************************************
 * @brief	    Detect touch key handler
 * @details     
 * @return      
 * @exception   No
 * @note            
 *******************************************************************************
 */
void API_DetectTouchKey(void)
{
    uint16_t BSP_TouchKeyHandle_Tmp;
    uint16_t BSP_TouchKeyHandle_ChangeKeyStatus;
    uint16_t BSP_TouchKeyHandle_Cmp;
    
    /*Scan TPad0 & TPad0*/
    if( MG04_06_CTR.Touchkey_State == TOUCHKEY_KEY0_CHARGE)
    {
        TK_CONF.Scan_Key   = Scan_PAD0;
        MG04_06_CTR.TouchKey_NowStatus = 0;
        API_TouchKeyCharge_Cmd( ENABLE);
    }
    else if( MG04_06_CTR.Touchkey_State == TOUCHKEY_TPAD01_SAMPLE)
    {
        MG04_06_CTR.TouchKey_NowStatus = API_TouchKey_GetStatus();
        API_TouchKeyCharge_Cmd( DISABLE);
    }
    /*Scan TPad2 & TPad3*/
    else if( MG04_06_CTR.Touchkey_State == TOUCHKEY_KEY1_CHARGE)
    {
        TK_CONF.Scan_Key = Scan_PAD1;
        API_TouchKeyCharge_Cmd( ENABLE);
    }
    else
    {
        MG04_06_CTR.TouchKey_NowStatus |= API_TouchKey_GetStatus(); 
        MG04_06_CTR.Touchkey_State = TOUCHKEY_KEY0_CHARGE;
        API_TouchKeyCharge_Cmd( DISABLE);
        
        if( MG04_06_CTR.TouchKey_NowStatus != MG04_06_CTR.TouchKey_BeforeStatus)
        {
            BSP_TouchKeyHandle_ChangeKeyStatus = MG04_06_CTR.TouchKey_NowStatus ^ MG04_06_CTR.TouchKey_BeforeStatus;
            BSP_TouchKeyHandle_ChangeKeyStatus = BSP_TouchKeyHandle_ChangeKeyStatus & MG04_06_CTR.TouchKey_NowStatus;   /*!< Only handle boutton make*/
            BSP_TouchKeyHandle_Cmp             = TK_KEY0F;
            
            for( BSP_TouchKeyHandle_Tmp = 0; BSP_TouchKeyHandle_Tmp< MG04_06_TOTAL_TPKEY; BSP_TouchKeyHandle_Tmp++)
            {
                if( (BSP_TouchKeyHandle_ChangeKeyStatus & BSP_TouchKeyHandle_Cmp)!=0)
                {
                    MG04_06_CTR.KeyStatus_ChangeFlag = 1;
                    MG04_06_CTR.KeyStatus[MG04_06_TPKEY_SHIFT + BSP_TouchKeyHandle_Tmp] =  MG04_06_KEY_FLAG;
                }
                
                BSP_TouchKeyHandle_Cmp = (uint16_t)(BSP_TouchKeyHandle_Cmp << 1);
            }
            
            MG04_06_CTR.TouchKey_BeforeStatus = MG04_06_CTR.TouchKey_NowStatus;
        }
        return;        
    }
    
    MG04_06_CTR.Touchkey_State = MG04_06_CTR.Touchkey_State + 1;
}
/**
 *******************************************************************************
 * @brief	    Detect SARADC key handler
 * @details     
 * @return      
 * @exception   No
 * @note            
 *******************************************************************************
 */
void API_DetectSARADCKey(void)
{
    uint32_t BSP_ScanKey_Tmp;
    uint32_t BSP_ScanKey_Cmp;
    
    MG04_06_CTR.SKey_NowStatus = BSP_SARADCKey_GetStatus();
    
    BSP_ScanKey_Cmp = ( MG04_06_CTR.SKey_NowStatus ^ MG04_06_CTR.SKey_BeforeStatus);
    
    if( BSP_ScanKey_Cmp!= 0)
    {
        MG04_06_CTR.SKey_BeforeStatus = MG04_06_CTR.SKey_NowStatus;
        
        for( BSP_ScanKey_Tmp = 0; BSP_ScanKey_Tmp < MG04_06_TOTAL_SKEY; BSP_ScanKey_Tmp++)
        {
            if( BSP_ScanKey_Cmp & 0x00000001)
            {
                MG04_06_CTR.KeyStatus[ BSP_ScanKey_Tmp] =  ((uint8_t)(MG04_06_CTR.SKey_NowStatus & 0x00000001)); 
                
                MG04_06_CTR.KeyStatus_ChangeFlag = 1;
            }
            
            MG04_06_CTR.SKey_NowStatus = MG04_06_CTR.SKey_NowStatus >> 1;
            BSP_ScanKey_Cmp = BSP_ScanKey_Cmp >> 1;
        }
    }
}

/**
 *******************************************************************************
 * @brief	    Up & Down key handler
 * @details     
 * @return      
 * @exception   No
 * @note            
 *******************************************************************************
 */
static void API_KeyFunction0_Handle(void)
{
    uint8_t API_KeyFunction0_HandleTmp ;
    
    API_KeyFunction0_HandleTmp = BK_UP | BK_DOWN;
    
    if( API_KeyFunction0_HandleTmp==0)
    {
        return;   
    }

    /*Normal key function*/
    if( MG04_06_CTR.SubStateLock == DISABLE)
    {
        if( BK_UP == 1)
        {
            BK_UP = 0;
            
            MG04_06_CTR.SubStateBefore = MG04_06_CTR.SubState;
            
            if( MG04_06_CTR.SubState ==0)
            {
                MG04_06_CTR.SubState = (MG04_06_ControlStateTotal[ MG04_06_CTR.MainState] - 1);
            }
            else
            {
                MG04_06_CTR.SubState = MG04_06_CTR.SubState - 1;
            }
            MG04_06_CTR.LCD_DynamicUpdateFlag |= LCD_DYNAMICUPDATE_OPITIONFRAME;
        }
        if( BK_DOWN == 1)
        {
            BK_DOWN = 0;
            
            MG04_06_CTR.SubStateBefore = MG04_06_CTR.SubState;
            
            if( MG04_06_CTR.SubState == (MG04_06_ControlStateTotal[ MG04_06_CTR.MainState] - 1)) 
            {
                MG04_06_CTR.SubState = 0;
            }
            else
            {
                MG04_06_CTR.SubState = MG04_06_CTR.SubState + 1;
            }
            MG04_06_CTR.LCD_DynamicUpdateFlag |= LCD_DYNAMICUPDATE_OPITIONFRAME;
        }
    
        return;
    }
    /*Special function in different Sub. Function*/
    switch( MG04_06_CTR.MainState)
    {
        case MG04_06_STATE_MAIN_SETTINGPAGE:
                                            if( MG04_06_CTR.SubState == MG04_06_STATE_SUB_IDLETIME)
                                            {
                                                if( BK_UP == 1)    
                                                {
                                                    if( MG04_06_CTR.IdleTimeCompare !=0)
                                                    {
                                                        MG04_06_CTR.IdleTimeCompare = MG04_06_CTR.IdleTimeCompare - 1000;
                                                    }
                                                    
                                                }
                                                if( BK_DOWN == 1)    
                                                {
                                                    if( MG04_06_CTR.IdleTimeCompare != 6000)
                                                    {
                                                        MG04_06_CTR.IdleTimeCompare = MG04_06_CTR.IdleTimeCompare + 1000;
                                                    }
                                                }
                                                
                                                MG04_06_CTR.IdleState = MG04_06_IDLESTATE_NO;
                                                MG04_06_CTR.LCD_DynamicUpdateFlag |= LCD_DYNAMICUPDATE_OTHER;
                                            }
                                            break;
        default:
                                            break;
    } 
}

/**
 *******************************************************************************
 * @brief	    ESC and Enter Key handler
 * @details     
 * @return      
 * @exception   No
 * @note            
 *******************************************************************************
 */
static void API_KeyFunction1_Handle(void)
{
    uint8_t API_KeyFunction1_HandleTmp;

    API_KeyFunction1_HandleTmp = BK_ESC | BK_ENTER;

    if( API_KeyFunction1_HandleTmp == 0)
    {
        return;
    }
    /*ESC Key Handle*/
    if( BK_ESC == 1)
    {
        BK_ESC = 0;
        
        MG04_06_CTR.ESC_FunctionFlag = 1;
    }
    /*Enter Key Handle*/
    if( BK_ENTER == 1)
    {
        BK_ENTER = 0;
        
        switch( MG04_06_CTR.pLCDEnterTable[ MG04_06_CTR.SubState])
        {
            case MG04_06_FUNCTION_SETTINGLOCK:
                                               MG04_06_CTR.SubStateLock = ENABLE;
                                               MG04_06_CTR.LCD_DynamicUpdateFlag |= LCD_DYNAMICUPDATE_OTHER;
                                               return;
            case MG04_06_FUNCTION_ESC:
                                               MG04_06_CTR.ESC_FunctionFlag = 1;
                                               return;
            case MG04_06_FUNCTION_ARGBMODE:
                                               API_ARGB_ChangeMode(MG04_06_CTR.SubState);
                                               return;
            case MG04_06_FUNCTION_ARGBCOLOR:
                                               API_ARGB_ChangeColor();
                                               return;
            case MG04_06_FUNCTION_AUTO:
                                               if( MG04_06_CTR.ARGB_Auto == 0)
                                               {
                                                   API_ARGB_ChangeMode(ARGB_AUTO_MODE);
                                               }
                                               else
                                               {
                                                   API_ARGB_ChangeMode(ARGB_MODE_DISABLE);
                                               }
                                               MG04_06_CTR.LCD_DynamicUpdateFlag |= LCD_DYNAMICUPDATE_OTHER;
                                               break;
            default:
                                               break;
        }
        
        if(MG04_06_CTR.pLCDEnterTable[ MG04_06_CTR.SubState] & MG04_06_FUNCTION_NEXTPAGE)
        {
            API_KeyFunction1_HandleTmp = MG04_06_CTR.pLCDEnterTable[ MG04_06_CTR.SubState] & (~MG04_06_FUNCTION_NEXTPAGE);
            
            if( MG04_06_CTR.MainState == MG04_06_STATE_MAIN_ARGBFRONTPAGE && 
                ARGB_CTR.ColorNumber == 2 && MG04_06_CTR.ARGB_Auto == DISABLE &&
                API_KeyFunction1_HandleTmp == MG04_06_STATE_MAIN_ARGBCOLORPAGE)
            {
                MG04_06_CTR.MainState      = MG04_06_STATE_MAIN_ARGB2COLORPAGE;
            } 
            else if( MG04_06_CTR.MainState == MG04_06_STATE_MAIN_ARGB2COLORPAGE)
            {
                MG04_06_CTR.MainState       = API_KeyFunction1_HandleTmp;
                MG04_06_CTR.ARGBColor_Index = MG04_06_CTR.SubState; 
            }
            else
            {
                if( API_KeyFunction1_HandleTmp == MG04_06_STATE_MAIN_ARGBCOLORPAGE)
                {
                    if( MG04_06_CTR.ARGB_Auto == 1)
                    {
                        return;
                    }
                    if( (ARGB_CTR.DramaMode & ARGB_MODE_MASK)== ARGB_MODE_DISABLE)
                    {
                        return;
                    }
                }
                MG04_06_CTR.MainState       = API_KeyFunction1_HandleTmp;
                MG04_06_CTR.ARGBColor_Index = 0;
            }
            MG04_06_CTR.SubState        = 0;
            MG04_06_CTR.SubStateBefore  = 0;
            
            API_LCDDynamicFormChange(MG04_06_CTR.MainState);
    
            
            
            MG04_06_CTR.LCD_BackgroundUpdateFlag  = 1;
            MG04_06_CTR.LCD_DynamicUpdateFlag |= LCD_DYNAMICUPDATE_OPITIONFRAME;
            
            switch( MG04_06_CTR.MainState) 
            {
                case MG04_06_STATE_MAIN_ARGBFRONTPAGE:
                case MG04_06_STATE_MAIN_SETTINGPAGE:
                                                      MG04_06_CTR.LCD_DynamicUpdateFlag |= LCD_DYNAMICUPDATE_OTHER;
                                                      break;
                default:                              
                                                      break;
            }
            return;
        }
    }
}

/**
 *******************************************************************************
 * @brief	    ARGB mode key , ARGB color0 key , ARGB color1 and ARGB Auto key
 *              handler.
 * @details     
 * @return      
 * @exception   No
 * @note            
 *******************************************************************************
 */
static void API_KeyFunction2_Handle(void)
{
    uint8_t API_KeyFunction2_HandleTmp;
    uint8_t API_KeyFunction2_HandleTmp2;
    
    API_KeyFunction2_HandleTmp = TK_FUN_ARGBMODE | TK_FUN_AUTO | TK_FUN_COLOR0 | TK_FUN_COLOR1;
    
    if( API_KeyFunction2_HandleTmp == 0)
    {
        return;
    }
    
    if( TK_FUN_ARGBMODE == 1  || TK_FUN_AUTO == 1)
    {
        TK_FUN_ARGBMODE = 0;
        
        /*Backgroud Update if now isn't in ARGB mode page.*/
        if( MG04_06_CTR.MainState != MG04_06_STATE_MAIN_ARGBMODEPAGE || TK_FUN_AUTO == 1)
        {
            MG04_06_CTR.MainState  = MG04_06_STATE_MAIN_ARGBMODEPAGE;
            API_LCDDynamicFormChange(MG04_06_CTR.MainState);
            MG04_06_CTR.LCD_BackgroundUpdateFlag  = 1;
        }
        
        MG04_06_CTR.LCD_DynamicUpdateFlag |= LCD_DYNAMICUPDATE_OPITIONFRAME;
        
        /*Change ARGB Mode*/
        if( TK_FUN_AUTO == 1)
        {
            TK_FUN_AUTO = 0;
            
            MG04_06_CTR.SubStateBefore =  ARGB_AUTO_MODE;
            MG04_06_CTR.SubState       =  ARGB_AUTO_MODE;
        }
        else if( MG04_06_CTR.ARGB_Auto == 1)
        {
            MG04_06_CTR.SubStateBefore =  ARGB_AUTO_MODE;
            MG04_06_CTR.SubState       =  0;            
        }
        else
        {
            MG04_06_CTR.SubStateBefore = (uint8_t)((ARGB_CTR.DramaMode & ARGB_MODE_MASK)>>ARGB_MODE_SHIFT);
            MG04_06_CTR.SubState       = MG04_06_CTR.SubStateBefore + 1; 
        }
        API_ARGB_ChangeMode(MG04_06_CTR.SubState);
    }
    
    if( MG04_06_CTR.ARGB_Auto == ENABLE)
    {
        TK_FUN_COLOR0 = 0;
        TK_FUN_COLOR1 = 0;
        return;
    }
    
    if( TK_FUN_COLOR0 == 1)
    {
        TK_FUN_COLOR0 = 0;
        
        if( (ARGB_CTR.DramaMode & ARGB_MODE_MASK)== ARGB_MODE_DISABLE)
        {
            return;
        }
        /*Backgroud Update if now isn't in ARGB mode page.*/
        if( MG04_06_CTR.MainState != MG04_06_STATE_MAIN_ARGBCOLORPAGE || MG04_06_CTR.ARGBColor_Index!=0)
        {
            MG04_06_CTR.MainState  = MG04_06_STATE_MAIN_ARGBCOLORPAGE;
            API_LCDDynamicFormChange(MG04_06_CTR.MainState);
            MG04_06_CTR.LCD_BackgroundUpdateFlag  = 1;
        }
        MG04_06_CTR.LCD_DynamicUpdateFlag |= LCD_DYNAMICUPDATE_OPITIONFRAME;
        
        /*Change ARGB Color0*/
        MG04_06_CTR.ARGBColor_Index = 0;
        
        API_KeyFunction2_HandleTmp2 = (MG04_06_CTR.ARGBMode_Index * 2) + MG04_06_CTR.ARGBColor_Index;
        
        API_KeyFunction2_HandleTmp = (uint8_t)MG04_06_CTR.ARGBColor_Opition[API_KeyFunction2_HandleTmp2];
        
        MG04_06_CTR.SubStateBefore = API_KeyFunction2_HandleTmp;
        
        if( API_KeyFunction2_HandleTmp == (MG04_06_ControlStateTotal[MG04_06_STATE_MAIN_ARGBCOLORPAGE]-1))
        {
            MG04_06_CTR.SubState = 0;
        }
        else
        {
            MG04_06_CTR.SubState = MG04_06_CTR.SubStateBefore + 1;
        }
        API_ARGB_ChangeColor();
    }
    if( TK_FUN_COLOR1 == 1)
    {
        TK_FUN_COLOR1 = 0;
        
        if( ARGB_CTR.ColorNumber !=2)
        {
            return;
        }
        /*Backgroud Update if now isn't in ARGB mode page.*/
        if( MG04_06_CTR.MainState != MG04_06_STATE_MAIN_ARGBCOLORPAGE || MG04_06_CTR.ARGBColor_Index!=1)
        {
            MG04_06_CTR.MainState  = MG04_06_STATE_MAIN_ARGBCOLORPAGE;
            API_LCDDynamicFormChange(MG04_06_CTR.MainState);
            MG04_06_CTR.LCD_BackgroundUpdateFlag  = 1;
        }
        MG04_06_CTR.LCD_DynamicUpdateFlag |= LCD_DYNAMICUPDATE_OPITIONFRAME;
        
        /*Change ARGB Color0*/
        MG04_06_CTR.ARGBColor_Index = 1;
        
        API_KeyFunction2_HandleTmp2 = (MG04_06_CTR.ARGBMode_Index * 2) + MG04_06_CTR.ARGBColor_Index;
        
        API_KeyFunction2_HandleTmp = (uint8_t)MG04_06_CTR.ARGBColor_Opition[API_KeyFunction2_HandleTmp2];
        
        MG04_06_CTR.SubStateBefore = API_KeyFunction2_HandleTmp;
        
        if( API_KeyFunction2_HandleTmp == (MG04_06_ControlStateTotal[MG04_06_STATE_MAIN_ARGBCOLORPAGE]-1))
        {
            MG04_06_CTR.SubState = 0;
        }
        else
        {
            MG04_06_CTR.SubState = MG04_06_CTR.SubStateBefore + 1;
        }
        
        API_ARGB_ChangeColor();
    }
}

/**
 *******************************************************************************
 * @brief	    ESC event handler
 * @details     
 * @return      
 * @exception   No
 * @note            
 *******************************************************************************
 */
static void API_ESCFunction_Handle(void)
{
    if( MG04_06_CTR.ESC_FunctionFlag == 0)
    {
        return;
    }
    
    MG04_06_CTR.ESC_FunctionFlag = 0;

    if( MG04_06_ESCDrama[ MG04_06_CTR.MainState] != 0xFA)
    {
        if(MG04_06_CTR.SubStateLock == ENABLE)
        {
            MG04_06_CTR.SubStateLock = DISABLE;

            MG04_06_CTR.LCD_DynamicUpdateFlag |= LCD_DYNAMICUPDATE_OTHER;
        }
        else
        {
            if( MG04_06_CTR.MainState == MG04_06_STATE_MAIN_ARGBCOLORPAGE && 
                ARGB_CTR.ColorNumber == 2)
            {
                MG04_06_CTR.MainState      = MG04_06_STATE_MAIN_ARGB2COLORPAGE;
            }
            else
            {
                MG04_06_CTR.MainState      = MG04_06_ESCDrama[ MG04_06_CTR.MainState];
            }

            MG04_06_CTR.SubState       = 0;
            MG04_06_CTR.SubStateBefore = 0;
            
            MG04_06_CTR.LCD_BackgroundUpdateFlag  = 1;
            MG04_06_CTR.LCD_DynamicUpdateFlag |= (LCD_DYNAMICUPDATE_OPITIONFRAME | LCD_DYNAMICUPDATE_OTHER);
            
            API_LCDDynamicFormChange(MG04_06_CTR.MainState);
        }
    }
}

/**
 *******************************************************************************
 * @brief	    Detect that press key handler 
 * @details     
 * @return      
 * @exception   No
 * @note            
 *******************************************************************************
 */
void API_Key_Handle(void)
{
    if( MG04_06_CTR.KeyStatus_ChangeFlag == 1)
    {
        MG04_06_CTR.KeyStatus_ChangeFlag = 0;
        
        /* LCD Idle wakeup*/
        if( MG04_06_CTR.IdleState == MG04_06_IDLESTATE_SLEEP)
        {
            //Clear All key status buffer. 
            memset(MG04_06_CTR.KeyStatus,0,MG04_06_TOTAL_KEY);

            MG04_06_CTR.IdleState = MG04_06_IDLESTATE_WAKEUP;
        }
        else
        {
            MG04_06_CTR.IdleState = MG04_06_IDLESTATE_NO;
        }
        
        MG04_06_CTR.IdleTime = 0;
        
        /* +, - ,Key Handle */
        API_KeyFunction0_Handle();
        /* ESC ,Enter Key Handle*/
        API_KeyFunction1_Handle();
        /*ARGB Function, ARGB Color0, ARGB Color1, ARGB Next Setting Handle*/
        API_KeyFunction2_Handle();
        /*ESC Function Handle*/
        API_ESCFunction_Handle();
    }
}

/**
 *******************************************************************************
 * @brief	    LCD idle handler
 * @details     
 * @return      
 * @exception   No
 * @note            
 *******************************************************************************
 */
void API_IdleHandle(void)
{
    if( MG04_06_CTR.IdleTimeCompare == 0)
    {
        return;
    }
    
    switch( MG04_06_CTR.IdleState)
    {
        case MG04_06_IDLESTATE_NO:
        case MG04_06_IDLESTATE_FORNTPAGE:
                                         break;
        case MG04_06_IDLESTATE_SLEEP:    return;
        case MG04_06_IDLESTATE_TOSLEEP:
                                         MG04_06_CTR.IdleState = MG04_06_IDLESTATE_SLEEP;
                                         __SPI_LCD_BACKLIGHT_OFF();
                                         return;
        case MG04_06_IDLESTATE_WAKEUP:
                                         MG04_06_CTR.IdleState = MG04_06_IDLESTATE_NO;
                                         __SPI_LCD_BACKLIGHT_ON();
                                         return;
        default:
                                         MG04_06_CTR.IdleState = MG04_06_IDLESTATE_NO;
                                         MG04_06_CTR.IdleTime  = 0;
                                         break;
    }
    
    MG04_06_CTR.IdleTime ++;
    
    if( MG04_06_CTR.IdleTime > (MG04_06_CTR.IdleTimeCompare + 50))
    {
        MG04_06_CTR.IdleState = MG04_06_IDLESTATE_TOSLEEP;
    }
    else if( MG04_06_CTR.IdleTime > MG04_06_CTR.IdleTimeCompare)
    {
        if( MG04_06_CTR.MainState != MG04_06_STATE_MAIN_FRONTPAGE)
        {
            MG04_06_CTR.MainState       = MG04_06_STATE_MAIN_FRONTPAGE;
            API_LCDDynamicFormChange(MG04_06_CTR.MainState);
            MG04_06_CTR.LCD_BackgroundUpdateFlag  = 1;
            MG04_06_CTR.SubStateBefore  = 0;
            MG04_06_CTR.SubState        = 0;
            MG04_06_CTR.LCD_DynamicUpdateFlag |= LCD_DYNAMICUPDATE_OPITIONFRAME;
        }
        else if( MG04_06_CTR.SubState !=0)
        {
            MG04_06_CTR.SubStateBefore  = MG04_06_CTR.SubState;
            MG04_06_CTR.SubState        = 0;
            MG04_06_CTR.LCD_DynamicUpdateFlag |= LCD_DYNAMICUPDATE_OPITIONFRAME;
        }
        
        MG04_06_CTR.SubStateLock = DISABLE;
        MG04_06_CTR.IdleState    = MG04_06_IDLESTATE_FORNTPAGE;
    }
}



///@}









